<?php
ob_start(); 
header('Content-Type: application/json');
$response = ['success' => false, 'message' => 'Gagal mendapatkan respons AI.', 'ai_response' => null, 'action_taken' => null];

if (file_exists(__DIR__ . '/../../config/config.php')) {
    require_once __DIR__ . '/../../config/config.php';
}
require_once __DIR__ . '/../../includes/session.php'; 
require_once __DIR__ . '/../../lib/Auth.php';
require_once __DIR__ . '/../../includes/db_connect.php';
require_once __DIR__ . '/../../includes/functions.php';

if (!Auth::isLoggedIn()) {
    http_response_code(401);
    $response['message'] = 'Sesi berakhir. Silakan login kembali.';
    $strayOutput = ob_get_clean(); if (!empty($strayOutput)) error_log("Stray output (pre-auth): " . $strayOutput);
    echo json_encode($response);
    exit;
}

if (!isset($pdo) || !$pdo instanceof PDO) {
    http_response_code(500);
    error_log("PDO object not available in get_ai_response.php");
    $response['message'] = 'Koneksi database gagal.';
    $strayOutput = ob_get_clean(); if (!empty($strayOutput)) error_log("Stray output (pre-pdo): " . $strayOutput);
    echo json_encode($response);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    $response['message'] = 'Metode request tidak valid.';
    $strayOutput = ob_get_clean(); if (!empty($strayOutput)) error_log("Stray output (pre-method): " . $strayOutput);
    echo json_encode($response);
    exit;
}

$userInput = trim($_POST['user_message'] ?? '');
$userId = $_SESSION['user_id'];
$userDisplayName = $_SESSION['user_fullname'] ?? $_SESSION['username'] ?? "Pengguna {$userId}";
if (empty(trim($userDisplayName)) || $userDisplayName === "Pengguna {$userId}") {
    try {
        $stmtUser = $pdo->prepare("SELECT full_name, username FROM users WHERE id = :user_id LIMIT 1");
        $stmtUser->bindParam(':user_id', $userId, PDO::PARAM_INT);
        $stmtUser->execute();
        $userData = $stmtUser->fetch(PDO::FETCH_ASSOC);
        if ($userData) {
            $userDisplayName = !empty(trim($userData['full_name'])) ? $userData['full_name'] : $userData['username'];
        }
    } catch (PDOException $e) {
        error_log("Error fetching user display name: " . $e->getMessage());
    }
}
$userDisplayName = htmlspecialchars($userDisplayName);

$appName = $appSettings['app_name'] ?? 'SMM Panel Kami';
$geminiApiKey = $appSettings['gemini_api_key'] ?? 'AIzaSyBRGtCf4UmikhrMLyyTx7wxu8LaK_HY5aQ';

if (empty($geminiApiKey)) {
    http_response_code(500);
    $response['message'] = 'Kunci API untuk layanan AI belum dikonfigurasi.';
    error_log("Gemini API Key is not configured.");
    $strayOutput = ob_get_clean(); if (!empty($strayOutput)) error_log("Stray output (pre-apikey): " . $strayOutput);
    echo json_encode($response);
    exit;
}

if (empty($userInput)) {
    http_response_code(400);
    $response['message'] = 'Pesan pengguna tidak boleh kosong.';
    $strayOutput = ob_get_clean(); if (!empty($strayOutput)) error_log("Stray output (pre-input): " . $strayOutput);
    echo json_encode($response);
    exit;
}

if (!isset($_SESSION['chat_ai_context'])) {
    $_SESSION['chat_ai_context'] = [
        'last_ai_question_type' => null,
        'pending_action_type' => null,
        'pending_order_id' => null,
        'history' => []
    ];
}
$chatContext = &$_SESSION['chat_ai_context'];

function formatLocalDatePhp(string $dateString = null, string $format = 'dd MMM yy HH:mm:ss'): string {
    if (empty($dateString)) {
        return 'N/A';
    }
    try {
        $dateTime = new DateTime($dateString);
        if (class_exists('IntlDateFormatter')) {
            $formatter = new IntlDateFormatter('id_ID', IntlDateFormatter::FULL, IntlDateFormatter::FULL, $dateTime->getTimezone()->getName(), IntlDateFormatter::GREGORIAN, $format);
            if ($formatter) {
                return $formatter->format($dateTime);
            }
        }
        $phpDateFormat = 'd M Y, H:i';
        if ($format === 'dd MMM yy HH:mm:ss') $phpDateFormat = 'd M y H:i:s';
        return date($phpDateFormat, $dateTime->getTimestamp());
    } catch (Exception $e) {
        error_log("Error formatting date '{$dateString}': " . $e->getMessage());
        return $dateString;
    }
}

function cleanAiResponseTags(string $rawResponse): string {
    return trim(preg_replace('/`?\[INTENT:[^\]]*\]`?/i', '', $rawResponse));
}

function callGeminiAPI(string $prompt, string $apiKey) {
    $geminiApiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-1.5-flash-latest:generateContent?key=' . $apiKey;
    $requestPayload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'safetySettings' => [
            ['category' => 'HARM_CATEGORY_HARASSMENT', 'threshold' => 'BLOCK_MEDIUM_AND_ABOVE'],
            ['category' => 'HARM_CATEGORY_HATE_SPEECH', 'threshold' => 'BLOCK_MEDIUM_AND_ABOVE'],
            ['category' => 'HARM_CATEGORY_SEXUALLY_EXPLICIT', 'threshold' => 'BLOCK_MEDIUM_AND_ABOVE'],
            ['category' => 'HARM_CATEGORY_DANGEROUS_CONTENT', 'threshold' => 'BLOCK_MEDIUM_AND_ABOVE'],
        ],
        'generationConfig' => ['temperature' => 0.3, 'maxOutputTokens' => 700]
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $geminiApiUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($requestPayload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_TIMEOUT, 50);
    $apiResultJson = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($curlError) {
        error_log("AI API cURL Error: " . $curlError);
        return ['success' => false, 'message' => 'Gagal menghubungi layanan AI: ' . $curlError, 'http_code' => $httpCode];
    }
    if ($httpCode !== 200) {
        $apiErrorDetails = json_decode($apiResultJson, true);
        $googleErrorMessage = $apiErrorDetails['error']['message'] ?? ('Status ' . $httpCode);
        error_log("AI API HTTP Error {$httpCode}: " . $apiResultJson);
        return ['success' => false, 'message' => 'Layanan AI mengembalikan error: ' . $googleErrorMessage, 'http_code' => $httpCode, 'raw_response' => $apiResultJson];
    }

    $apiResult = json_decode($apiResultJson, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("AI API JSON Decode Error. Response: " . $apiResultJson);
        return ['success' => false, 'message' => 'Gagal memproses respons dari AI (JSON tidak valid).'];
    }
    if (isset($apiResult['candidates'][0]['content']['parts'][0]['text'])) {
        return ['success' => true, 'ai_response' => $apiResult['candidates'][0]['content']['parts'][0]['text']];
    }
    if (isset($apiResult['promptFeedback']['blockReason'])) {
        $blockReason = $apiResult['promptFeedback']['blockReason'];
        error_log("AI API Safety Block: " . $blockReason . " | Input: " . substr($prompt, -200));
        return ['success' => false, 'ai_response' => "Maaf, saya tidak bisa merespons permintaan Anda saat ini karena alasan keamanan ({$blockReason}). Silakan coba pertanyaan lain atau hubungi admin."];
    }
    error_log("AI API Unrecognized Response Format or No Candidate: " . $apiResultJson);
    return ['success' => false, 'message' => 'Format respons dari AI tidak dikenali.', 'raw_response' => $apiResultJson];
}

function validateOrderForUser(int $orderId, int $currentUserId, PDO $db): array|false {
    try {
        $stmt = $db->prepare("SELECT o.id, o.status, s.name as service_name, o.target, o.quantity, o.price, o.created_at
                              FROM orders o
                              JOIN services s ON o.service_id = s.id
                              WHERE o.id = :order_id AND o.user_id = :user_id LIMIT 1");
        $stmt->bindParam(':order_id', $orderId, PDO::PARAM_INT);
        $stmt->bindParam(':user_id', $currentUserId, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("DB Error validating order: " . $e->getMessage());
        return false;
    }
}

function createTicketForUser(int $currentUserId, string $subject, string $message, PDO $db): string|false {
    try {
        $ticketId = 'TKT-' . strtoupper(substr(md5(uniqid(rand(), true)), 0, 10));
        $db->beginTransaction();

        $sqlInsertTicket = "INSERT INTO tickets (ticket_id, user_id, subject, message, status, priority, last_reply_by, is_read_user, is_read_admin, created_at, updated_at)
                            VALUES (:ticket_id, :user_id, :subject, :message, 'open', 'medium', 'user', 1, 0, NOW(), NOW())";
        $stmtInsertTicket = $db->prepare($sqlInsertTicket);
        $stmtInsertTicket->bindParam(':ticket_id', $ticketId);
        $stmtInsertTicket->bindParam(':user_id', $currentUserId, PDO::PARAM_INT);
        $stmtInsertTicket->bindParam(':subject', $subject);
        $stmtInsertTicket->bindParam(':message', $message);
        if (!$stmtInsertTicket->execute()) {
            $db->rollBack(); return false;
        }
        $newTicketInternalId = $db->lastInsertId();

        $sqlInsertReply = "INSERT INTO ticket_replies (ticket_id, user_id, message, created_at)
                           VALUES (:ticket_id, :user_id, :message, NOW())";
        $stmtInsertReply = $db->prepare($sqlInsertReply);
        $stmtInsertReply->bindParam(':ticket_id', $newTicketInternalId, PDO::PARAM_INT);
        $stmtInsertReply->bindParam(':user_id', $currentUserId, PDO::PARAM_INT);
        $stmtInsertReply->bindParam(':message', $message);
        if (!$stmtInsertReply->execute()) {
            $db->rollBack(); return false;
        }

        $db->commit();
        return $ticketId;
    } catch (PDOException $e) {
        if ($db->inTransaction()) $db->rollBack();
        error_log("DB Error creating ticket: " . $e->getMessage());
        return false;
    }
}

function getServiceRecommendations(array $categoryKeywords, array $criteria, PDO $db): array {
    if (empty($categoryKeywords)) {
        return [];
    }
    $categoryPlaceholders = [];
    $categoryParams = [];
    $categoryWhereClauses = [];
    foreach ($categoryKeywords as $index => $keyword) {
        $placeholder = ":cat_keyword_" . $index;
        $categoryPlaceholders[] = $placeholder;
        $categoryParams[$placeholder] = "%" . trim($keyword) . "%";
        $categoryWhereClauses[] = "c.name LIKE " . $placeholder;
    }
    $categoryWhereSql = implode(" OR ", $categoryWhereClauses);
    $sqlFindCategories = "SELECT c.id FROM categories c WHERE ({$categoryWhereSql}) AND c.status = 'active'";
    $stmtFindCategories = $db->prepare($sqlFindCategories);
    $stmtFindCategories->execute($categoryParams);
    $categoryIds = $stmtFindCategories->fetchAll(PDO::FETCH_COLUMN);

    if (empty($categoryIds)) {
        return [];
    }
    $categoryIdsPlaceholder = implode(',', array_fill(0, count($categoryIds), '?'));
    $orderByClause = "s.price_per_1000 ASC";
    $sqlServices = "SELECT s.id, s.name, s.price_per_1000, s.min_order, s.max_order, s.note, c.name as category_name
                    FROM services s
                    JOIN categories c ON s.category_id = c.id
                    WHERE s.category_id IN ({$categoryIdsPlaceholder}) AND s.status = 'active'
                    ORDER BY {$orderByClause}
                    LIMIT 3";
    $stmtServices = $db->prepare($sqlServices);
    $stmtServices->execute($categoryIds);
    return $stmtServices->fetchAll(PDO::FETCH_ASSOC);
}

$chatContext['history'][] = ['role' => 'user', 'content' => $userInput];
if (count($chatContext['history']) > 6) {
    $chatContext['history'] = array_slice($chatContext['history'], -6);
}

$directlyProcessed = false;
$orderDetails = null;
$actionType = null;
$orderId = null;
$orderIdArray = [];

if ($chatContext['last_ai_question_type'] === 'ASKED_FOR_ORDER_ID_FOR_ACTION' && $chatContext['pending_action_type'] && preg_match_all('/\d+/', $userInput, $idMatchesAll)) {
    if (!empty($idMatchesAll[0])) {
        $orderIdArray = array_map('intval', $idMatchesAll[0]);
        $actionType = $chatContext['pending_action_type'];
        if (count($orderIdArray) === 1) {
            $orderId = $orderIdArray[0];
            $orderDetails = validateOrderForUser($orderId, $userId, $pdo);
        }
        $directlyProcessed = true;
    }
} elseif ($chatContext['last_ai_question_type'] === 'ASKED_FOR_ACTION_FOR_ORDER_ID' && $chatContext['pending_order_id'] && preg_match('/^(status|cek status|cancel|batal|batalkan|refill|isi ulang)$/i', $userInput, $actionMatches)) {
    $orderId = $chatContext['pending_order_id'];
    $orderIdArray = [$orderId];
    $rawAction = strtolower($actionMatches[1]);
    if (in_array($rawAction, ['status', 'cek status'])) $actionType = 'STATUS';
    elseif (in_array($rawAction, ['cancel', 'batal', 'batalkan'])) $actionType = 'CANCEL';
    elseif (in_array($rawAction, ['refill', 'isi ulang'])) $actionType = 'REFILL';

    if ($actionType) {
        $orderDetails = validateOrderForUser($orderId, $userId, $pdo);
        $directlyProcessed = true;
    }
}

if ($directlyProcessed) {
    if (count($orderIdArray) > 1) {
        $response['action_taken'] = 'VALIDATE_MULTIPLE_ORDERS_AND_PROCESS';
        $response['order_ids_to_process'] = $orderIdArray;
        $response['action_type_to_process'] = $actionType;
        $processedResults = [];
        foreach ($orderIdArray as $singleOrderId) {
            $currentOrderDetails = validateOrderForUser($singleOrderId, $userId, $pdo);
            if (!$currentOrderDetails) {
                $processedResults[] = "Pesanan #{$singleOrderId}: Tidak ditemukan atau bukan milik Anda.";
                continue;
            }
            if ($actionType === 'STATUS') {
                $processedResults[] = "Pesanan #{$singleOrderId} (" . htmlspecialchars($currentOrderDetails['service_name']) . "): Status saat ini '" . htmlspecialchars($currentOrderDetails['status']) . "'.";
            } elseif ($actionType === 'CANCEL') {
                $cancellableStatuses = ['pending', 'processing'];
                if (in_array(strtolower($currentOrderDetails['status']), $cancellableStatuses)) {
                    $ticketSubject = "Permintaan Pembatalan Pesanan #{$singleOrderId} (via AI Chat) - Pengguna: {$userDisplayName}";
                    $ticketMessage = "Pengguna {$appName} ({$userDisplayName} - User ID: {$userId}) meminta pembatalan untuk pesanan ID: {$singleOrderId}.\nDetail: {$currentOrderDetails['service_name']}, Target: {$currentOrderDetails['target']}.\nStatus Saat Ini: {$currentOrderDetails['status']}.";
                    $newTicketId = createTicketForUser($userId, $ticketSubject, $ticketMessage, $pdo);
                    $processedResults[] = $newTicketId ? "Pesanan #{$singleOrderId}: Permintaan pembatalan diteruskan via tiket #{$newTicketId}." : "Pesanan #{$singleOrderId}: Gagal membuat tiket pembatalan.";
                } else {
                    $processedResults[] = "Pesanan #{$singleOrderId}: Tidak dapat dibatalkan (Status: " . htmlspecialchars($currentOrderDetails['status']) . ").";
                }
            } elseif ($actionType === 'REFILL') {
                $refillableStatuses = ['success', 'partial'];
                if (in_array(strtolower($currentOrderDetails['status']), $refillableStatuses)) {
                    $ticketSubject = "Permintaan Refill Pesanan #{$singleOrderId} (via AI Chat) - Pengguna: {$userDisplayName}";
                    $ticketMessage = "Pengguna {$appName} ({$userDisplayName} - User ID: {$userId}) meminta refill untuk pesanan ID: {$singleOrderId}.\nDetail: {$currentOrderDetails['service_name']}, Target: {$currentOrderDetails['target']}.\nStatus Saat Ini: {$currentOrderDetails['status']}.";
                    $newTicketId = createTicketForUser($userId, $ticketSubject, $ticketMessage, $pdo);
                    $processedResults[] = $newTicketId ? "Pesanan #{$singleOrderId}: Permintaan refill diteruskan via tiket #{$newTicketId}." : "Pesanan #{$singleOrderId}: Gagal membuat tiket refill.";
                } else {
                    $processedResults[] = "Pesanan #{$singleOrderId}: Tidak dapat di-refill (Status: " . htmlspecialchars($currentOrderDetails['status']) . ").";
                }
            }
        }
        $summaryMessage = "Berikut hasil pemrosesan untuk permintaan {$actionType} Anda:\n" . implode("\n", $processedResults);
        $followUpPromptForMultiple = "Anda telah memproses permintaan {$actionType} untuk beberapa pesanan dari {$userDisplayName}. Ringkasan hasilnya adalah sebagai berikut:\n{$summaryMessage}\n\nSampaikan ringkasan ini kepada pengguna dengan ramah dan jelas. Jangan meminta ID pesanan lagi untuk tindakan ini.";
        $geminiFollowUpResultMultiple = callGeminiAPI($followUpPromptForMultiple, $geminiApiKey);
        $response['ai_response'] = cleanAiResponseTags($geminiFollowUpResultMultiple['success'] ? $geminiFollowUpResultMultiple['ai_response'] : $summaryMessage);

    } elseif ($orderId && $actionType) {
        $response['action_taken'] = 'VALIDATE_ORDER_AND_PROCESS';
        $response['order_id_to_process'] = $orderId;
        $response['action_type_to_process'] = $actionType;
        if (!$orderDetails) {
            $followUpPrompt = "Pengguna {$userDisplayName} memberikan ID Pesanan: {$orderId} untuk tindakan {$actionType}, namun ID tersebut tidak valid atau bukan miliknya. Sampaikan ini dan minta ID yang benar. Jangan proses tindakan {$actionType} ini.";
            $geminiFollowUpResult = callGeminiAPI($followUpPrompt, $geminiApiKey);
            $response['ai_response'] = cleanAiResponseTags($geminiFollowUpResult['success'] ? $geminiFollowUpResult['ai_response'] : "Maaf {$userDisplayName}, ID Pesanan #{$orderId} tidak ditemukan atau bukan milik Anda. Mohon berikan ID yang benar.");
            $chatContext['last_ai_question_type'] = 'ASKED_FOR_ORDER_ID_FOR_ACTION';
            $chatContext['pending_action_type'] = $actionType;
            $chatContext['pending_order_id'] = null;
        } else {
            if ($actionType === 'STATUS') {
                $statusText = htmlspecialchars($orderDetails['status'] ?? 'Tidak diketahui');
                $serviceNameText = htmlspecialchars($orderDetails['service_name'] ?? 'Layanan');
                $targetText = htmlspecialchars($orderDetails['target'] ?? '-');
                $quantityText = number_format($orderDetails['quantity'] ?? 0);
                $createdDateText = htmlspecialchars(formatLocalDatePhp($orderDetails['created_at'] ?? ''));
                $followUpPrompt = "Pengguna {$userDisplayName} bertanya status pesanan #{$orderId}. Pesanan valid. Detailnya: Layanan '{$serviceNameText}', Target '{$targetText}', Jumlah {$quantityText}, Status saat ini '{$statusText}', Dibuat pada {$createdDateText}. Sampaikan informasi ini kepada pengguna {$userDisplayName} dengan ramah. Jangan meminta ID pesanan lagi untuk ini.";
                $geminiFollowUpResult = callGeminiAPI($followUpPrompt, $geminiApiKey);
                $response['ai_response'] = cleanAiResponseTags($geminiFollowUpResult['success'] ? $geminiFollowUpResult['ai_response'] : "Status pesanan #{$orderId} Anda adalah '{$statusText}'.");
            } elseif ($actionType === 'CANCEL') {
                $cancellableStatuses = ['pending', 'processing'];
                if (in_array(strtolower($orderDetails['status']), $cancellableStatuses)) {
                    $ticketSubject = "Permintaan Pembatalan Pesanan #{$orderId} (via AI Chat) - Pengguna: {$userDisplayName}";
                    $ticketMessage = "Pengguna {$appName} ({$userDisplayName} - User ID: {$userId}) meminta pembatalan untuk pesanan ID: {$orderId}.\nDetail Pesanan:\nLayanan: {$orderDetails['service_name']}\nTarget: {$orderDetails['target']}\nJumlah: {$orderDetails['quantity']}\nStatus Saat Ini: {$orderDetails['status']}\n\nPesan Asli Pengguna:\n{$userInput}";
                    $newTicketId = createTicketForUser($userId, $ticketSubject, $ticketMessage, $pdo);
                    $followUpPrompt = $newTicketId ? "Konfirmasi kepada pengguna {$userDisplayName} bahwa tiket dukungan #{$newTicketId} telah berhasil dibuat untuk memproses permintaan pembatalan pesanannya dengan ID #{$orderId}. Informasikan juga bahwa tim admin akan segera menindaklanjuti tiket tersebut. Jangan meminta informasi tambahan seperti ID pesanan lagi." : "Informasikan kepada {$userDisplayName} bahwa gagal membuat tiket dukungan untuk pembatalan pesanan #{$orderId} karena masalah teknis. Mohon maaf atas ketidaknyamanannya dan sarankan pengguna untuk mencoba membuat tiket manual atau menghubungi admin langsung.";
                } else {
                    $followUpPrompt = "Informasikan kepada {$userDisplayName} bahwa pesanan #{$orderId} miliknya saat ini berstatus '{$orderDetails['status']}' dan tidak dapat dibatalkan melalui sistem otomatis. Sarankan untuk menghubungi dukungan jika ada pertanyaan lebih lanjut. Jangan meminta ID pesanan lagi.";
                }
                $geminiFollowUpResult = callGeminiAPI($followUpPrompt, $geminiApiKey);
                $response['ai_response'] = cleanAiResponseTags($geminiFollowUpResult['success'] ? $geminiFollowUpResult['ai_response'] : "Status pesanan #{$orderId} Anda adalah '{$orderDetails['status']}' dan mungkin tidak dapat dibatalkan.");
            } elseif ($actionType === 'REFILL') {
                $refillableStatuses = ['success', 'partial'];
                if (in_array(strtolower($orderDetails['status']), $refillableStatuses) ) {
                    $ticketSubject = "Permintaan Refill Pesanan #{$orderId} (via AI Chat) - Pengguna: {$userDisplayName}";
                    $ticketMessage = "Pengguna {$appName} ({$userDisplayName} - User ID: {$userId}) meminta refill untuk pesanan ID: {$orderId}.\nDetail Pesanan:\nLayanan: {$orderDetails['service_name']}\nTarget: {$orderDetails['target']}\nJumlah: {$orderDetails['quantity']}\nStatus Saat Ini: {$orderDetails['status']}\n\nPesan Asli Pengguna:\n{$userInput}";
                    $newTicketId = createTicketForUser($userId, $ticketSubject, $ticketMessage, $pdo);
                    $followUpPrompt = $newTicketId ? "Konfirmasi kepada pengguna {$userDisplayName} bahwa tiket dukungan #{$newTicketId} telah berhasil dibuat untuk memproses permintaan isi ulang (refill) pesanannya dengan ID #{$orderId}. Informasikan juga bahwa tim admin akan segera menindaklanjuti tiket tersebut. Jangan meminta informasi tambahan seperti ID pesanan lagi." : "Informasikan kepada {$userDisplayName} bahwa gagal membuat tiket dukungan untuk refill pesanan #{$orderId} karena masalah teknis. Mohon maaf atas ketidaknyamanannya dan sarankan pengguna untuk mencoba membuat tiket manual atau menghubungi admin langsung.";
                } else {
                    $followUpPrompt = "Informasikan kepada {$userDisplayName} bahwa pesanan #{$orderId} miliknya saat ini berstatus '{$orderDetails['status']}' dan tidak dapat di-refill melalui sistem otomatis saat ini, atau layanan ini mungkin tidak mendukung refill. Sarankan untuk menghubungi dukungan jika ada pertanyaan lebih lanjut. Jangan meminta ID pesanan lagi.";
                }
                $geminiFollowUpResult = callGeminiAPI($followUpPrompt, $geminiApiKey);
                $response['ai_response'] = cleanAiResponseTags($geminiFollowUpResult['success'] ? $geminiFollowUpResult['ai_response'] : "Status pesanan #{$orderId} Anda adalah '{$orderDetails['status']}' dan mungkin tidak dapat di-refill.");
            }
            $chatContext['last_ai_question_type'] = null;
            $chatContext['pending_action_type'] = null;
            $chatContext['pending_order_id'] = null;
        }
    }
    $response['success'] = true;
} else {
    $historyString = "";
    if (isset($chatContext['history']) && is_array($chatContext['history'])) {
        foreach (array_slice($chatContext['history'], -4) as $turn) {
            if (isset($turn['role']) && isset($turn['content'])) {
                $historyString .= ($turn['role'] === 'user' ? "Pengguna ({$userDisplayName}): " : "AI: ") . cleanAiResponseTags($turn['content']) . "\n";
            }
        }
    }

    $initialPromptContext = "
Anda adalah \"Asisten Virtual {$appName}\", chatbot AI yang sangat ramah, sopan, dan profesional untuk platform SMM Panel \"{$appName}\".
Pengguna saat ini adalah: {$userDisplayName}.
Riwayat singkat percakapan terakhir (jika ada, untuk konteks Anda):
{$historyString}
Tugas utama Anda adalah menjawab pertanyaan pengguna HANYA yang berkaitan dengan penggunaan, fitur, dan informasi umum dari sisi PENGGUNA.

PENANGANAN KHUSUS UNTUK KOMPLAIN ATAU PERTANYAAN TERKAIT PESANAN SPESIFIK:
1.  Jika pengguna bertanya tentang status pesanan, ingin membatalkan pesanan, atau meminta refill pesanan, TAPI TIDAK MENYERTAKAN ID PESANAN:
    Minta pengguna untuk menyebutkan ID Pesanan terlebih dahulu.
    Contoh respons: \"Tentu {$userDisplayName}, bisa bantu sebutkan ID Pesanan yang Anda maksud agar saya bisa membantu lebih lanjut?\"
    Sertakan tag: `[INTENT:GET_ORDER_ID_FOR_ACTION,ACTION_TYPE:STATUS/CANCEL/REFILL]` (sesuaikan ACTION_TYPE).
2.  Jika pengguna memberikan satu atau LEBIH ID Pesanan untuk dicek statusnya, dibatalkan, atau direfill (misal 'cek status 123 dan 456' atau 'batalkan pesanan 10, 11, 12'):
    Sertakan tag: `[INTENT:PROCESS_MULTIPLE_ORDER_ACTIONS,ACTION_TYPE:STATUS/CANCEL/REFILL,ORDER_IDS:id1,id2,id3]` (pisahkan ID dengan koma).
    Contoh respons: \"Baik {$userDisplayName}, saya akan coba proses permintaan {$actionType} untuk pesanan #[daftar ID]. `[INTENT:PROCESS_MULTIPLE_ORDER_ACTIONS,ACTION_TYPE:{$actionType},ORDER_IDS:{$daftarID}]`\"
3.  Jika pengguna memberikan ID Pesanan tanpa maksud yang jelas, dan Anda tidak yakin (setelah melihat riwayat singkat):
    Tanyakan maksudnya. Contoh: \"Baik {$userDisplayName}, untuk pesanan ID [id_pesanan_dari_user], apakah Anda ingin mengecek status, membatalkan, atau meminta refill?\"
    Sertakan tag: `[INTENT:GET_ACTION_FOR_ORDER_ID,ORDER_ID:id_pesanan_dari_user]`

PENANGANAN PERMINTAAN REKOMENDASI LAYANAN:
1. Jika pengguna meminta rekomendasi layanan, coba ekstrak:
   a. KATA KUNCI KATEGORI: Ini bisa berupa platform (Instagram, TikTok, YouTube, Facebook, Twitter, Telegram, Shopee, Spotify, dll.) dan/atau jenis layanan (followers, like, views, komentar, subscriber, dll.). Jika pengguna menyebutkan \"instagram followers\", maka kata kunci kategorinya adalah \"instagram,followers\". Jika hanya \"instagram\", kata kuncinya \"instagram\". Jika hanya \"followers\", kata kuncinya \"followers\".
   b. KRITERIA: Kata-kata seperti \"murah\", \"termurah\", \"populer\", \"terbaik\", \"paling banyak dipesan\", \"cepat\", \"instant\".
2. Jika KATA KUNCI KATEGORI berhasil diekstrak (minimal satu kata kunci platform atau jenis layanan yang dikenali):
   Sertakan tag: `[INTENT:RECOMMEND_SERVICES,CATEGORY_KEYWORDS:kata_kunci_platform,kata_kunci_jenis_layanan,CRITERIA:kriteria1,kriteria2]`
   Contoh: \"harga layanan instagram followers termurah\" -> `[INTENT:RECOMMEND_SERVICES,CATEGORY_KEYWORDS:instagram,followers,CRITERIA:termurah]`
   Contoh: \"follower tiktok yang populer\" -> `[INTENT:RECOMMEND_SERVICES,CATEGORY_KEYWORDS:tiktok,follower,CRITERIA:populer]`
   Contoh: \"layanan IG views\" -> `[INTENT:RECOMMEND_SERVICES,CATEGORY_KEYWORDS:instagram,views,CRITERIA:]` (Kriteria bisa kosong)
3. Jika pengguna hanya menyebutkan kriteria (misal \"layanan termurah\") TANPA menyebutkan KATA KUNCI KATEGORI yang jelas (platform atau jenis layanan):
   Minta klarifikasi kategori/platform kepada pengguna.
   Contoh respons: \"Tentu {$userDisplayName}, untuk merekomendasikan layanan termurah, layanan untuk platform apa (misalnya Instagram, TikTok, dll.) dan jenis apa (followers, likes, views, dll.) yang Anda maksud?\"
   PENTING: Dalam respons klarifikasi ini, JANGAN sertakan tag `[INTENT:...]` apapun. Biarkan sistem memproses input pengguna berikutnya.

JANGAN PERNAH mencoba melakukan aksi (cancel/refill/memberi status) sendiri. Tugas Anda hanya mengidentifikasi permintaan dan ID, lalu sistem akan memprosesnya.
Jika pengguna bertanya tentang hal di luar topik, tolak dengan sopan dan sarankan untuk membuat tiket ke admin jika relevan.
Selalu jawab dalam Bahasa Indonesia yang baik dan format jawaban menggunakan Markdown sederhana (hindari tabel, gunakan daftar poin).
Identifikasi diri sebagai \"Asisten Virtual {$appName}\". Jawaban singkat, padat, informatif.
";
    $currentPrompt = "{$initialPromptContext}\n\nPertanyaan Pengguna Saat Ini: {$userInput}";
    $geminiResult = callGeminiAPI($currentPrompt, $geminiApiKey);

    if (!$geminiResult['success']) {
        $response['message'] = $geminiResult['message'] ?? 'Gagal mendapatkan respons dari AI.';
        if (isset($geminiResult['raw_response'])) $response['debug_ai_raw'] = $geminiResult['raw_response'];
        http_response_code(isset($geminiResult['http_code']) && $geminiResult['http_code'] >= 400 ? $geminiResult['http_code'] : 500);
    } else {
        $aiRawResponse = $geminiResult['ai_response'];
        $response['ai_response'] = cleanAiResponseTags($aiRawResponse);
        $response['success'] = true;

        if (preg_match('/\[INTENT:GET_ORDER_ID_FOR_ACTION,ACTION_TYPE:([A-Z\/]+)\]/i', $aiRawResponse, $matchesIntentNoId)) {
            $response['action_taken'] = 'ASK_FOR_ORDER_ID';
            $response['next_action_type'] = strtoupper($matchesIntentNoId[1]);
            $chatContext['last_ai_question_type'] = 'ASKED_FOR_ORDER_ID_FOR_ACTION';
            $chatContext['pending_action_type'] = strtoupper($matchesIntentNoId[1]);
            $chatContext['pending_order_id'] = null;
        } elseif (preg_match('/\[INTENT:GET_ACTION_FOR_ORDER_ID,ORDER_ID:(\d+)\]/i', $aiRawResponse, $matchesIntentNoAction)) {
            $response['action_taken'] = 'ASK_FOR_ACTION';
            $response['order_id_to_clarify'] = (int)$matchesIntentNoAction[1];
            $chatContext['last_ai_question_type'] = 'ASKED_FOR_ACTION_FOR_ORDER_ID';
            $chatContext['pending_order_id'] = (int)$matchesIntentNoAction[1];
            $chatContext['pending_action_type'] = null;
        } elseif (preg_match('/\[INTENT:PROCESS_MULTIPLE_ORDER_ACTIONS,ACTION_TYPE:([A-Z\/]+),ORDER_IDS:([\d,]+)\]/i', $aiRawResponse, $matchesMultipleIds)) {
            $actionTypeMultiple = strtoupper($matchesMultipleIds[1]);
            $orderIdsString = $matchesMultipleIds[2];
            $orderIdArrayMultiple = array_map('intval', explode(',', $orderIdsString));

            $response['action_taken'] = 'VALIDATE_MULTIPLE_ORDERS_AND_PROCESS';
            $response['order_ids_to_process'] = $orderIdArrayMultiple;
            $response['action_type_to_process'] = $actionTypeMultiple;

            if ($actionTypeMultiple === 'CANCEL' || $actionTypeMultiple === 'REFILL' || $actionTypeMultiple === 'STATUS') {
                $processedResultsMultiple = [];
                foreach ($orderIdArrayMultiple as $singleOrderIdMultiple) {
                    $orderDetailsLoop = validateOrderForUser($singleOrderIdMultiple, $userId, $pdo);
                    if (!$orderDetailsLoop) {
                        $processedResultsMultiple[] = "Pesanan #{$singleOrderIdMultiple}: Tidak ditemukan atau bukan milik Anda.";
                        continue;
                    }
                    if ($actionTypeMultiple === 'STATUS') {
                         $processedResultsMultiple[] = "Pesanan #{$singleOrderIdMultiple} (" . htmlspecialchars($orderDetailsLoop['service_name']) . "): Status '" . htmlspecialchars($orderDetailsLoop['status']) . "'.";
                    } elseif ($actionTypeMultiple === 'CANCEL') {
                        $cancellableStatuses = ['pending', 'processing'];
                        if (in_array(strtolower($orderDetailsLoop['status']), $cancellableStatuses)) {
                            $ticketSubject = "Permintaan Pembatalan Pesanan #{$singleOrderIdMultiple} (via AI Chat) - Pengguna: {$userDisplayName}";
                            $ticketMessage = "Pengguna {$appName} ({$userDisplayName} - User ID: {$userId}) meminta pembatalan untuk pesanan ID: {$singleOrderIdMultiple}.\nDetail: {$orderDetailsLoop['service_name']}, Target: {$orderDetailsLoop['target']}.\nStatus Saat Ini: {$orderDetailsLoop['status']}.";
                            $newTicketId = createTicketForUser($userId, $ticketSubject, $ticketMessage, $pdo);
                            $processedResultsMultiple[] = $newTicketId ? "Pesanan #{$singleOrderIdMultiple}: Permintaan pembatalan diteruskan via tiket #{$newTicketId}." : "Pesanan #{$singleOrderIdMultiple}: Gagal membuat tiket pembatalan.";
                        } else {
                            $processedResultsMultiple[] = "Pesanan #{$singleOrderIdMultiple}: Tidak dapat dibatalkan (Status: " . htmlspecialchars($orderDetailsLoop['status']) . ").";
                        }
                    } elseif ($actionTypeMultiple === 'REFILL') {
                        $refillableStatuses = ['success', 'partial'];
                        if (in_array(strtolower($orderDetailsLoop['status']), $refillableStatuses)) {
                            $ticketSubject = "Permintaan Refill Pesanan #{$singleOrderIdMultiple} (via AI Chat) - Pengguna: {$userDisplayName}";
                            $ticketMessage = "Pengguna {$appName} ({$userDisplayName} - User ID: {$userId}) meminta refill untuk pesanan ID: {$singleOrderIdMultiple}.\nDetail: {$orderDetailsLoop['service_name']}, Target: {$orderDetailsLoop['target']}.\nStatus Saat Ini: {$orderDetailsLoop['status']}.";
                            $newTicketId = createTicketForUser($userId, $ticketSubject, $ticketMessage, $pdo);
                            $processedResultsMultiple[] = $newTicketId ? "Pesanan #{$singleOrderIdMultiple}: Permintaan refill diteruskan via tiket #{$newTicketId}." : "Pesanan #{$singleOrderIdMultiple}: Gagal membuat tiket refill.";
                        } else {
                            $processedResultsMultiple[] = "Pesanan #{$singleOrderIdMultiple}: Tidak dapat di-refill (Status: " . htmlspecialchars($orderDetailsLoop['status']) . ").";
                        }
                    }
                }
                $summaryMessageMultiple = "Berikut hasil pemrosesan untuk permintaan {$actionTypeMultiple} Anda:\n" . implode("\n", $processedResultsMultiple);
                $followUpPromptForMultipleAI = "Anda telah memproses permintaan {$actionTypeMultiple} untuk beberapa pesanan dari {$userDisplayName}. Ringkasan hasilnya adalah sebagai berikut:\n{$summaryMessageMultiple}\n\nSampaikan ringkasan ini kepada pengguna dengan ramah dan jelas menggunakan format daftar (bullet points), bukan tabel. Jangan meminta ID pesanan lagi untuk tindakan ini.";
                $geminiFollowUpResultMultipleAI = callGeminiAPI($followUpPromptForMultipleAI, $geminiApiKey);
                $response['ai_response'] = cleanAiResponseTags($geminiFollowUpResultMultipleAI['success'] ? $geminiFollowUpResultMultipleAI['ai_response'] : $summaryMessageMultiple);

                $chatContext['last_ai_question_type'] = null;
                $chatContext['pending_action_type'] = null;
                $chatContext['pending_order_id'] = null;
            }
        } elseif (preg_match('/\[INTENT:RECOMMEND_SERVICES,CATEGORY_KEYWORDS:([^,\]]+(?:,[^,\]]+)*)(?:,CRITERIA:([^,\]]+(?:,[^,\]]+)*))?\]/i', $aiRawResponse, $matchesRecommend)) {
            $categoryKeywordsString = $matchesRecommend[1] ?? '';
            $criteriaString = $matchesRecommend[3] ?? '';

            $categoryKeywords = !empty($categoryKeywordsString) ? array_map('trim', explode(',', $categoryKeywordsString)) : [];
            $criteria = !empty($criteriaString) ? array_map('trim', explode(',', $criteriaString)) : [];
            $response['action_taken'] = 'RECOMMEND_SERVICES';

            if (empty($categoryKeywords)) {
                $followUpPromptRecommend = "Pengguna {$userDisplayName} meminta rekomendasi layanan dengan kriteria \"" . implode(", ", $criteria) . "\" tetapi tidak menyebutkan kategori/platform. Minta pengguna untuk menyebutkan platform (misalnya Instagram, TikTok, dll.) dan jenis layanan (followers, likes, dll.). PENTING: Jangan sertakan tag [INTENT] apapun dalam respons klarifikasi ini.";
            } else {
                $recommendedServices = getServiceRecommendations($categoryKeywords, $criteria, $pdo);
                if (!empty($recommendedServices)) {
                    $serviceDetailsText = "Tentu, {$userDisplayName}! Berikut beberapa rekomendasi layanan " . htmlspecialchars(implode(" & ", $categoryKeywords)) . " yang mungkin Anda suka (diurutkan berdasarkan harga termurah):\n\n";
                    foreach ($recommendedServices as $idx => $service) {
                        $serviceDetailsText .= "**" . ($idx + 1) . ". " . htmlspecialchars($service['name']) . "** (ID: `" . $service['id'] . "`)\n";
                        $serviceDetailsText .= "   - Kategori: " . htmlspecialchars($service['category_name']) . "\n";
                        $serviceDetailsText .= "   - Harga: Rp " . number_format($service['price_per_1000'], 0, ',', '.') . " /1000\n";
                        $serviceDetailsText .= "   - Min/Max Pesan: " . number_format($service['min_order']) . " / " . number_format($service['max_order']) . "\n";
                        if (!empty($service['note'])) {
                            $noteCleaned = str_replace(["\r\n", "\r", "\n"], ' ', $service['note']);
                            $serviceDetailsText .= "   - Catatan: " . htmlspecialchars(substr($noteCleaned, 0, 100)) . (strlen($noteCleaned) > 100 ? "..." : "") . "\n";
                        }
                        $serviceDetailsText .= "\n";
                    }
                    $serviceDetailsText .= "Anda bisa langsung memesan dengan menyebutkan ID layanan yang diinginkan, atau lihat daftar layanan lengkap di menu Layanan.";
                    $followUpPromptRecommend = "Pengguna {$userDisplayName} meminta rekomendasi layanan. Berikut adalah layanan yang ditemukan (sudah diformat sebagai daftar, bukan tabel):\n{$serviceDetailsText}\nSampaikan ini kepada pengguna. Pastikan untuk tidak menyertakan tag [INTENT] apapun dalam respons akhirmu dan gunakan format daftar yang diberikan.";
                } else {
                    $followUpPromptRecommend = "Pengguna {$userDisplayName} meminta rekomendasi layanan dengan kata kunci kategori: \"" . implode(", ", $categoryKeywords) . "\" dan kriteria: \"" . implode(", ", $criteria) . "\". Namun, tidak ada layanan yang cocok ditemukan. Informasikan pengguna dengan sopan dan sarankan untuk mencoba kata kunci lain atau melihat daftar layanan lengkap. Jangan sertakan tag [INTENT] apapun dalam respons akhirmu.";
                }
            }
            $geminiFollowUpResultRecommend = callGeminiAPI($followUpPromptRecommend, $geminiApiKey);
            $response['ai_response'] = cleanAiResponseTags($geminiFollowUpResultRecommend['success'] ? $geminiFollowUpResultRecommend['ai_response'] : "Maaf, saya tidak menemukan layanan yang cocok saat ini atau butuh info platform lebih spesifik.");

            $chatContext['last_ai_question_type'] = null;
            $chatContext['pending_action_type'] = null;
            $chatContext['pending_order_id'] = null;
        } else {
            $chatContext['last_ai_question_type'] = null;
            $chatContext['pending_action_type'] = null;
            $chatContext['pending_order_id'] = null;
        }
    }
}

$chatContext['history'][] = ['role' => 'assistant', 'content' => $response['ai_response']];
if (count($chatContext['history']) > 6) {
    $chatContext['history'] = array_slice($chatContext['history'], -6);
}

$strayOutput = ob_get_clean();
if (!empty($strayOutput)) {
    error_log("Stray output detected in get_ai_response.php: " . $strayOutput);
    if (!$response['success'] || !isset($response['ai_response']) || $response['ai_response'] === null) {
        $response['success'] = false;
        $response['message'] = 'Terjadi kesalahan internal server (PHP Error). Silakan coba lagi nanti.';
        $response['ai_response'] = null;
        if (defined('DEBUG_MODE') && DEBUG_MODE === true) {
            if (!isset($response['debug'])) $response['debug'] = [];
            $response['debug']['stray_php_output'] = $strayOutput;
        }
    }
}

if (!headers_sent()) {
    header('Content-Type: application/json');
}
echo json_encode($response);
exit;
?>
