<?php
    header('Content-Type: application/json');
    $response = ['success' => false, 'informations' => [], 'pagination' => null, 'message' => 'Gagal memuat data informasi.'];

    require_once __DIR__ . '/../../../config/config.php';
    require_once __DIR__ . '/../../../includes/session.php';
    require_once __DIR__ . '/../../../lib/Auth.php';
    require_once __DIR__ . '/../../../includes/functions.php';

    if (!Auth::isLoggedIn() || $_SESSION['user_role'] !== 'admin') {
        http_response_code(403);
        $response['message'] = 'Akses ditolak.';
        echo json_encode($response);
        exit;
    }

    if (!isset($pdo) || !$pdo instanceof PDO) {
        http_response_code(500);
        error_log("PDO object not available in get_admin_informations.php");
        $response['message'] = 'Koneksi database gagal.';
        echo json_encode($response);
        exit;
    }

    try {
        $limit = filter_input(INPUT_GET, 'limit', FILTER_VALIDATE_INT, ['options' => ['default' => 15, 'min_range' => 5, 'max_range' => 100]]);
        $page = filter_input(INPUT_GET, 'page', FILTER_VALIDATE_INT, ['options' => ['default' => 1, 'min_range' => 1]]);
        $status = $_GET['status'] ?? 'all';
        $sortBy = $_GET['sort_by'] ?? 'created_at';
        $sortType = $_GET['sort_type'] ?? 'DESC';
        $searchColumn = $_GET['search_column'] ?? 'id';
        $searchKeyword = filter_input(INPUT_GET, 'search_keyword', FILTER_SANITIZE_SPECIAL_CHARS) ?? '';

        $offset = ($page - 1) * $limit;

        $allowedSortColumns = ['created_at', 'updated_at', 'id', 'title', 'status'];
        $allowedSortTypes = ['ASC', 'DESC'];
        if (!in_array($sortBy, $allowedSortColumns)) { $sortBy = 'created_at'; }
        if (!in_array(strtoupper($sortType), $allowedSortTypes)) { $sortType = 'DESC'; }

        $allowedSearchColumns = ['id', 'title', 'content'];
        if (!in_array($searchColumn, $allowedSearchColumns)) { $searchColumn = 'id'; }

        $sqlSelect = "SELECT id, title, content, image_filename, status, created_at, updated_at ";
        $sqlFrom = "FROM informations ";
        $sqlWhereBase = "WHERE 1=1 ";
        $params = [];
        $whereClauses = [];

        if ($status !== 'all') {
            $validStatuses = ['active', 'inactive'];
            if (in_array($status, $validStatuses)) {
                $whereClauses[] = "status = :status";
                $params[':status'] = $status;
            }
        }

        if (!empty($searchKeyword)) {
            if ($searchColumn === 'id') {
                if (is_numeric($searchKeyword)) {
                    $whereClauses[] = "id = :search_keyword_id";
                    $params[':search_keyword_id'] = (int)$searchKeyword;
                } else {
                    $whereClauses[] = "1 = 0";
                }
            } else {
                $whereClauses[] = $searchColumn . " LIKE :search_keyword";
                $params[':search_keyword'] = '%' . $searchKeyword . '%';
            }
        }

        $sqlWhere = $sqlWhereBase . (!empty($whereClauses) ? " AND " . implode(" AND ", $whereClauses) : "");

        $totalItems = 0;
        $totalPages = 1;
        $sqlCount = "SELECT COUNT(id) " . $sqlFrom . $sqlWhere;
        try {
            $stmtCount = $pdo->prepare($sqlCount);
            $stmtCount->execute($params);
            $totalItems = (int)$stmtCount->fetchColumn();
            $totalPages = $totalItems > 0 ? ceil($totalItems / $limit) : 1;
            if ($page > $totalPages) {
                $page = $totalPages;
                $offset = ($page - 1) * $limit;
            } elseif ($page < 1) {
                $page = 1;
                $offset = 0;
            }
        } catch (\PDOException $e) {
            error_log("Error counting admin informations: " . $e->getMessage());
            throw $e;
        }

        $informations = [];
        $sqlData = $sqlSelect . $sqlFrom . $sqlWhere . " ORDER BY {$sortBy} {$sortType} LIMIT :limit OFFSET :offset";
        try {
            $stmtData = $pdo->prepare($sqlData);
            foreach ($params as $key => &$value) {
                 if ($key === ':search_keyword_id') {
                     $stmtData->bindValue($key, $value, PDO::PARAM_INT);
                 } else {
                     $stmtData->bindValue($key, $value);
                 }
            }
            unset($value);

            $stmtData->bindParam(':limit', $limit, PDO::PARAM_INT);
            $stmtData->bindParam(':offset', $offset, PDO::PARAM_INT);

            $stmtData->execute();
            $informations = $stmtData->fetchAll(PDO::FETCH_ASSOC);

            foreach ($informations as &$info) {
                $info['image_url'] = $info['image_filename'] ? base_url('uploads/informations/' . $info['image_filename']) : null;
                $info['content_snippet'] = mb_substr(strip_tags($info['content'] ?? ''), 0, 100);
                 if (mb_strlen(strip_tags($info['content'] ?? '')) > 100) {
                     $info['content_snippet'] .= '...';
                 }
            }
            unset($info);

            $response['success'] = true;
            $response['informations'] = $informations;
            $response['pagination'] = [
                'totalItems' => $totalItems,
                'totalPages' => $totalPages,
                'currentPage' => $page,
                'limit' => $limit,
                'offset' => $offset
            ];
            unset($response['message']);

        } catch (\PDOException $e) {
            error_log("Error fetching admin informations data: " . $e->getMessage());
            $response['message'] = "Gagal mengambil data informasi.";
            http_response_code(500);
        }

    } catch (\Throwable $e) {
        error_log("AJAX Get Admin Informations General Error: " . $e->getMessage());
        $response['message'] = "Terjadi kesalahan internal server.";
        if (defined('DEBUG_MODE') && DEBUG_MODE === true) { $response['debug_php'] = $e->getMessage(); }
        http_response_code(500);
    }

    echo json_encode($response);
    exit;
    ?>
    