<?php
header('Content-Type: application/json');
$response = ['success' => false, 'deposits' => [], 'pagination' => null, 'message' => 'Gagal memuat data deposit.'];

require_once __DIR__ . '/../../../config/config.php';
require_once __DIR__ . '/../../../includes/session.php';
require_once __DIR__ . '/../../../lib/Auth.php';
require_once __DIR__ . '/../../../includes/functions.php';

if (!Auth::isLoggedIn() || $_SESSION['user_role'] !== 'admin') {
    http_response_code(403);
    $response['message'] = 'Akses ditolak.';
    echo json_encode($response);
    exit;
}

if (!isset($pdo) || !$pdo instanceof PDO) {
    http_response_code(500);
    error_log("PDO object not available in get_admin_deposits.php");
    $response['message'] = 'Koneksi database gagal.';
    echo json_encode($response);
    exit;
}

try {
    $limit = filter_input(INPUT_GET, 'limit', FILTER_VALIDATE_INT, ['options' => ['default' => 15, 'min_range' => 5, 'max_range' => 100]]);
    $page = filter_input(INPUT_GET, 'page', FILTER_VALIDATE_INT, ['options' => ['default' => 1, 'min_range' => 1]]);
    $status = $_GET['status'] ?? 'all';
    $sortBy = $_GET['sort_by'] ?? 'd.created_at';
    $sortType = $_GET['sort_type'] ?? 'DESC';
    $searchColumn = $_GET['search_column'] ?? 'd.deposit_id';
    $searchKeyword = filter_input(INPUT_GET, 'search_keyword', FILTER_SANITIZE_SPECIAL_CHARS) ?? '';
    $dateStart = $_GET['date_start'] ?? '';
    $dateEnd = $_GET['date_end'] ?? '';
    $userIdFilter = filter_input(INPUT_GET, 'user_id', FILTER_VALIDATE_INT);
    $methodIdFilter = filter_input(INPUT_GET, 'method_id', FILTER_VALIDATE_INT);

    $offset = ($page - 1) * $limit;

    $allowedSortColumns = ['d.created_at', 'd.updated_at', 'd.deposit_id', 'u.username', 'dm.name', 'd.amount', 'd.status'];
    if (!in_array($sortBy, $allowedSortColumns)) {
        $sortBy = 'd.created_at';
    }
    $allowedSortTypes = ['ASC', 'DESC'];
    if (!in_array(strtoupper($sortType), $allowedSortTypes)) {
        $sortType = 'DESC';
    }

    $allowedSearchColumns = ['d.deposit_id', 'd.paydisini_unique_code', 'u.username', 'u.id', 'dm.name'];
     if (!in_array($searchColumn, $allowedSearchColumns)) {
        $searchColumn = 'd.deposit_id';
    }

    $sqlSelect = "SELECT d.id, d.deposit_id, d.user_id, d.paydisini_unique_code, d.amount, d.bonus, d.received_amount, d.status, d.created_at, d.updated_at, 
                         u.username as user_username, dm.name as method_name, dm.logo_url as method_logo ";
    $sqlFrom = "FROM deposits d 
                JOIN users u ON d.user_id = u.id 
                JOIN deposit_methods dm ON d.deposit_method_id = dm.id ";
    $sqlWhereBase = "WHERE 1=1 ";
    $params = [];
    $whereClauses = [];

    if ($status !== 'all') {
        $validStatuses = ['pending', 'success', 'failed', 'expired', 'canceled'];
        if (in_array($status, $validStatuses)) {
            $whereClauses[] = "d.status = :status";
            $params[':status'] = $status;
        }
    }

    if ($userIdFilter) {
        $whereClauses[] = "d.user_id = :user_id_filter";
        $params[':user_id_filter'] = $userIdFilter;
    }
    if ($methodIdFilter) {
        $whereClauses[] = "d.deposit_method_id = :method_id_filter";
        $params[':method_id_filter'] = $methodIdFilter;
    }

    if (!empty($searchKeyword)) {
        if ($searchColumn === 'u.id') {
            if (is_numeric($searchKeyword)) {
                $whereClauses[] = "u.id = :search_keyword_id";
                $params[':search_keyword_id'] = (int)$searchKeyword;
            } else {
                $whereClauses[] = "1 = 0"; 
            }
        } else {
            $whereClauses[] = $searchColumn . " LIKE :search_keyword";
            $params[':search_keyword'] = '%' . $searchKeyword . '%';
        }
    }

    if (!empty($dateStart) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $dateStart)) {
        $whereClauses[] = "DATE(d.created_at) >= :date_start";
        $params[':date_start'] = $dateStart;
    }
    if (!empty($dateEnd) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $dateEnd)) {
        $whereClauses[] = "DATE(d.created_at) <= :date_end";
        $params[':date_end'] = $dateEnd;
    }

    $sqlWhere = $sqlWhereBase . (!empty($whereClauses) ? " AND " . implode(" AND ", $whereClauses) : "");

    $totalItems = 0;
    $totalPages = 1;
    $sqlCount = "SELECT COUNT(d.id) " . $sqlFrom . $sqlWhere;
    $stmtCount = $pdo->prepare($sqlCount);
    $stmtCount->execute($params);
    $totalItems = (int)$stmtCount->fetchColumn();
    $totalPages = $totalItems > 0 ? ceil($totalItems / $limit) : 1;

    if ($page > $totalPages && $totalPages > 0) {
        $page = $totalPages;
        $offset = ($page - 1) * $limit;
    } elseif ($page < 1) {
        $page = 1;
        $offset = 0;
    }

    $deposits = [];
    $sqlData = $sqlSelect . $sqlFrom . $sqlWhere . " ORDER BY {$sortBy} {$sortType} LIMIT :limit OFFSET :offset";
    $stmtData = $pdo->prepare($sqlData);

    foreach ($params as $key => &$value) {
        if ($key === ':user_id_filter' || $key === ':method_id_filter' || $key === ':search_keyword_id') {
            $stmtData->bindValue($key, $value, PDO::PARAM_INT);
        } else {
            $stmtData->bindValue($key, $value);
        }
    }
    unset($value);

    $stmtData->bindParam(':limit', $limit, PDO::PARAM_INT);
    $stmtData->bindParam(':offset', $offset, PDO::PARAM_INT);
    $stmtData->execute();
    $deposits = $stmtData->fetchAll(PDO::FETCH_ASSOC);

    $response['success'] = true;
    $response['deposits'] = $deposits;
    $response['pagination'] = [
        'totalItems' => $totalItems,
        'totalPages' => $totalPages,
        'currentPage' => $page,
        'limit' => $limit,
        'offset' => $offset
    ];
    unset($response['message']);

} catch (\PDOException $e) {
    error_log("Admin Get Deposits DB Error: " . $e->getMessage());
    $response['message'] = "Terjadi kesalahan database saat memuat data deposit.";
    http_response_code(500);
} catch (\Throwable $e) {
    error_log("Admin Get Deposits General Error: " . $e->getMessage());
    $response['message'] = "Terjadi kesalahan internal server.";
    if (defined('DEBUG_MODE') && DEBUG_MODE === true) { $response['debug_php'] = $e->getMessage(); }
    http_response_code(500);
}

echo json_encode($response);
exit;
?>
